<?php
/**
 * License Validation API
 * Endpoint: https://license.strikesnake.com/api/validate.php
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config.php';

// Get request data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data) {
    $data = $_POST;
}

$license_key = isset($data['license_key']) ? trim($data['license_key']) : '';
$domain = isset($data['domain']) ? trim($data['domain']) : '';
$product = isset($data['product']) ? trim($data['product']) : 'elementor-page-builder-assistant';

// Get client IP
$client_ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown';

// Validate input
if (empty($license_key) || empty($domain)) {
    http_response_code(400);
    echo json_encode([
        'valid' => false,
        'message' => 'License key and domain are required'
    ]);
    exit;
}

try {
    // Find license
    $stmt = $pdo->prepare("
        SELECT * FROM licenses 
        WHERE license_key = ? AND product = ?
        LIMIT 1
    ");
    $stmt->execute([$license_key, $product]);
    $license = $stmt->fetch();

    if (!$license) {
        logActivity($pdo, 0, 'validation_failed', $client_ip, "Invalid key: $license_key");
        
        echo json_encode([
            'valid' => false,
            'message' => 'Invalid license key'
        ]);
        exit;
    }

    // Check if license is active
    if ($license['status'] !== 'active') {
        logActivity($pdo, $license['id'], 'validation_failed', $client_ip, "Status: {$license['status']}");
        
        echo json_encode([
            'valid' => false,
            'message' => 'License is ' . $license['status'],
            'status' => $license['status']
        ]);
        exit;
    }

    // Check expiration
    if ($license['expires_at'] && strtotime($license['expires_at']) < time()) {
        // Auto-expire
        $stmt = $pdo->prepare("UPDATE licenses SET status = 'expired' WHERE id = ?");
        $stmt->execute([$license['id']]);
        
        logActivity($pdo, $license['id'], 'validation_failed', $client_ip, 'License expired');
        
        echo json_encode([
            'valid' => false,
            'message' => 'License has expired',
            'expired_at' => $license['expires_at']
        ]);
        exit;
    }

    // Check domain activation
    if ($license['domain'] === null) {
        // First activation - save domain
        $stmt = $pdo->prepare("
            UPDATE licenses 
            SET domain = ?, current_activations = 1, updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$domain, $license['id']]);
        
        logActivity($pdo, $license['id'], 'first_activation', $client_ip, "Domain: $domain");
        
    } elseif ($license['domain'] !== $domain) {
        // Domain mismatch
        logActivity($pdo, $license['id'], 'validation_failed', $client_ip, "Domain mismatch: $domain vs {$license['domain']}");
        
        echo json_encode([
            'valid' => false,
            'message' => 'License is already activated on another domain',
            'activated_domain' => $license['domain']
        ]);
        exit;
    }

    // Success - log validation
    logActivity($pdo, $license['id'], 'validation_success', $client_ip, "Domain: $domain");

    // Update last check time
    $stmt = $pdo->prepare("UPDATE licenses SET updated_at = NOW() WHERE id = ?");
    $stmt->execute([$license['id']]);

    echo json_encode([
        'valid' => true,
        'message' => 'License is valid',
        'license_key' => $license_key,
        'customer_name' => $license['customer_name'],
        'expires_at' => $license['expires_at'],
        'status' => 'active',
        'domain' => $license['domain']
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'valid' => false,
        'message' => 'Server error: ' . $e->getMessage()
    ]);
}
