<?php
/**
 * License Deactivation API
 * Endpoint: https://license.strikesnake.com/api/deactivate.php
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config.php';

// Get request data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data) {
    $data = $_POST;
}

$license_key = isset($data['license_key']) ? trim($data['license_key']) : '';
$domain = isset($data['domain']) ? trim($data['domain']) : '';

// Get client IP
$client_ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown';

// Validate input
if (empty($license_key) || empty($domain)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'License key and domain are required'
    ]);
    exit;
}

try {
    // Find license
    $stmt = $pdo->prepare("
        SELECT * FROM licenses 
        WHERE license_key = ?
        LIMIT 1
    ");
    $stmt->execute([$license_key]);
    $license = $stmt->fetch();

    if (!$license) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid license key'
        ]);
        exit;
    }

    // Check if domain matches
    if ($license['domain'] !== $domain) {
        logActivity($pdo, $license['id'], 'deactivation_failed', $client_ip, "Domain mismatch: $domain vs {$license['domain']}");
        
        echo json_encode([
            'success' => false,
            'message' => 'Domain mismatch'
        ]);
        exit;
    }

    // Deactivate - clear domain
    $stmt = $pdo->prepare("
        UPDATE licenses 
        SET domain = NULL, current_activations = 0, updated_at = NOW() 
        WHERE id = ?
    ");
    $stmt->execute([$license['id']]);

    logActivity($pdo, $license['id'], 'deactivation_success', $client_ip, "Domain: $domain");

    echo json_encode([
        'success' => true,
        'message' => 'License deactivated successfully'
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Server error: ' . $e->getMessage()
    ]);
}
