<?php
/**
 * Admin Dashboard - License Management
 */

session_start();
require_once '../config.php';

// Check authentication
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit;
}

// Handle actions
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    // Create new license
    if ($action === 'create') {
        $license_key = generateLicenseKey();
        $customer_name = $_POST['customer_name'] ?? '';
        $customer_email = $_POST['customer_email'] ?? '';
        $customer_telegram = $_POST['customer_telegram'] ?? '';
        $expires_at = $_POST['expires_at'] ?? null;
        $notes = $_POST['notes'] ?? '';
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO licenses (license_key, customer_name, customer_email, customer_telegram, expires_at, notes) 
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$license_key, $customer_name, $customer_email, $customer_telegram, $expires_at ?: null, $notes]);
            $message = "License created: $license_key";
            $message_type = 'success';
        } catch (Exception $e) {
            $message = "Error: " . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    // Update license status
    if ($action === 'update_status') {
        $license_id = $_POST['license_id'] ?? 0;
        $status = $_POST['status'] ?? 'active';
        
        try {
            $stmt = $pdo->prepare("UPDATE licenses SET status = ? WHERE id = ?");
            $stmt->execute([$status, $license_id]);
            $message = "License status updated";
            $message_type = 'success';
        } catch (Exception $e) {
            $message = "Error: " . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    // Reset domain
    if ($action === 'reset_domain') {
        $license_id = $_POST['license_id'] ?? 0;
        
        try {
            $stmt = $pdo->prepare("UPDATE licenses SET domain = NULL, current_activations = 0 WHERE id = ?");
            $stmt->execute([$license_id]);
            $message = "Domain reset successfully";
            $message_type = 'success';
        } catch (Exception $e) {
            $message = "Error: " . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    // Delete license
    if ($action === 'delete') {
        $license_id = $_POST['license_id'] ?? 0;
        
        try {
            $stmt = $pdo->prepare("DELETE FROM licenses WHERE id = ?");
            $stmt->execute([$license_id]);
            $message = "License deleted";
            $message_type = 'success';
        } catch (Exception $e) {
            $message = "Error: " . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Get statistics
$stats = [
    'total' => $pdo->query("SELECT COUNT(*) FROM licenses")->fetchColumn(),
    'active' => $pdo->query("SELECT COUNT(*) FROM licenses WHERE status = 'active'")->fetchColumn(),
    'inactive' => $pdo->query("SELECT COUNT(*) FROM licenses WHERE status = 'inactive'")->fetchColumn(),
    'expired' => $pdo->query("SELECT COUNT(*) FROM licenses WHERE status = 'expired'")->fetchColumn(),
];

// Get all licenses
$stmt = $pdo->query("SELECT * FROM licenses ORDER BY created_at DESC");
$licenses = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>License Management - Admin Panel</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <div class="container">
        <header>
            <h1>🔐 License System</h1>
            <div class="header-actions">
                <span>Welcome, <?php echo htmlspecialchars($_SESSION['admin_username']); ?></span>
                <a href="logout.php" class="btn btn-sm btn-secondary">Logout</a>
            </div>
        </header>
        
        <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>
        
        <!-- Statistics -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['total']; ?></div>
                <div class="stat-label">Total Licenses</div>
            </div>
            <div class="stat-card stat-success">
                <div class="stat-value"><?php echo $stats['active']; ?></div>
                <div class="stat-label">Active</div>
            </div>
            <div class="stat-card stat-warning">
                <div class="stat-value"><?php echo $stats['inactive']; ?></div>
                <div class="stat-label">Inactive</div>
            </div>
            <div class="stat-card stat-danger">
                <div class="stat-value"><?php echo $stats['expired']; ?></div>
                <div class="stat-label">Expired</div>
            </div>
        </div>
        
        <!-- Create License Form -->
        <div class="card">
            <h2>Create New License</h2>
            <form method="POST" class="form-horizontal">
                <input type="hidden" name="action" value="create">
                <div class="form-row">
                    <div class="form-group">
                        <label>Customer Name</label>
                        <input type="text" name="customer_name" required>
                    </div>
                    <div class="form-group">
                        <label>Customer Email</label>
                        <input type="email" name="customer_email">
                    </div>
                    <div class="form-group">
                        <label>Telegram</label>
                        <input type="text" name="customer_telegram" placeholder="@username">
                    </div>
                    <div class="form-group">
                        <label>Expires At (Optional)</label>
                        <input type="date" name="expires_at">
                    </div>
                </div>
                <div class="form-group">
                    <label>Notes</label>
                    <textarea name="notes" rows="2"></textarea>
                </div>
                <button type="submit" class="btn btn-primary">Generate License</button>
            </form>
        </div>
        
        <!-- Licenses Table -->
        <div class="card">
            <h2>All Licenses (<?php echo count($licenses); ?>)</h2>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>License Key</th>
                            <th>Customer</th>
                            <th>Status</th>
                            <th>Domain</th>
                            <th>Created</th>
                            <th>Expires</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($licenses as $license): ?>
                        <tr>
                            <td>
                                <code class="license-key"><?php echo htmlspecialchars($license['license_key']); ?></code>
                                <button class="btn-copy" onclick="copyToClipboard('<?php echo $license['license_key']; ?>')">📋</button>
                            </td>
                            <td>
                                <strong><?php echo htmlspecialchars($license['customer_name'] ?: 'N/A'); ?></strong><br>
                                <small><?php echo htmlspecialchars($license['customer_email'] ?: ''); ?></small>
                                <?php if ($license['customer_telegram']): ?>
                                    <br><small>📱 <?php echo htmlspecialchars($license['customer_telegram']); ?></small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="badge badge-<?php echo $license['status']; ?>">
                                    <?php echo strtoupper($license['status']); ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($license['domain']): ?>
                                    <code><?php echo htmlspecialchars($license['domain']); ?></code>
                                <?php else: ?>
                                    <span class="text-muted">Not activated</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo date('Y-m-d', strtotime($license['created_at'])); ?></td>
                            <td>
                                <?php if ($license['expires_at']): ?>
                                    <?php echo date('Y-m-d', strtotime($license['expires_at'])); ?>
                                <?php else: ?>
                                    <span class="text-muted">Lifetime</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <form method="POST" style="display:inline;">
                                        <input type="hidden" name="action" value="update_status">
                                        <input type="hidden" name="license_id" value="<?php echo $license['id']; ?>">
                                        <select name="status" onchange="this.form.submit()" class="select-sm">
                                            <option value="active" <?php echo $license['status'] === 'active' ? 'selected' : ''; ?>>Active</option>
                                            <option value="inactive" <?php echo $license['status'] === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                                            <option value="suspended" <?php echo $license['status'] === 'suspended' ? 'selected' : ''; ?>>Suspended</option>
                                            <option value="expired" <?php echo $license['status'] === 'expired' ? 'selected' : ''; ?>>Expired</option>
                                        </select>
                                    </form>
                                    
                                    <?php if ($license['domain']): ?>
                                    <form method="POST" style="display:inline;" onsubmit="return confirm('Reset domain for this license?');">
                                        <input type="hidden" name="action" value="reset_domain">
                                        <input type="hidden" name="license_id" value="<?php echo $license['id']; ?>">
                                        <button type="submit" class="btn btn-sm btn-warning" title="Reset Domain">🔄</button>
                                    </form>
                                    <?php endif; ?>
                                    
                                    <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this license?');">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="license_id" value="<?php echo $license['id']; ?>">
                                        <button type="submit" class="btn btn-sm btn-danger" title="Delete">🗑️</button>
                                    </form>
                                    
                                    <a href="logs.php?license_id=<?php echo $license['id']; ?>" class="btn btn-sm btn-info" title="View Logs">📊</a>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <script>
        function copyToClipboard(text) {
            navigator.clipboard.writeText(text).then(() => {
                alert('License key copied to clipboard!');
            });
        }
    </script>
</body>
</html>
