# 🔐 License System - license.strikesnake.com

Professional license management system for Elementor Page Builder Assistant plugin.

## 📋 Features

- ✅ **License Generation** - Auto-generate secure license keys (XXXX-XXXX-XXXX-XXXX format)
- ✅ **API Validation** - Remote validation via REST API
- ✅ **Domain Binding** - One license per domain (auto-binds on first activation)
- ✅ **Status Management** - Active, Inactive, Suspended, Expired
- ✅ **Activity Logging** - Track all validation attempts and actions
- ✅ **Admin Panel** - Modern UI for managing licenses
- ✅ **Customer Management** - Store customer info, email, telegram
- ✅ **Expiration Dates** - Optional expiration or lifetime licenses
- ✅ **Security** - Secure hashing, session management, SQL injection protection

## 🚀 Installation

### 1. Database Setup

Create MySQL database:

```sql
CREATE DATABASE license_system CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

### 2. Configure

Edit `config.php`:

```php
define('DB_HOST', 'localhost');
define('DB_NAME', 'license_system');
define('DB_USER', 'your_db_user');
define('DB_PASS', 'your_db_password');

define('LICENSE_SALT', 'change_this_to_random_string');
define('API_SECRET_KEY', 'change_this_to_random_string');
```

### 3. Install Database Tables

Run installation:

```bash
php install.php
```

Or visit: `https://license.strikesnake.com/install.php`

This creates 3 tables:
- `licenses` - Store all license keys
- `license_logs` - Activity tracking
- `admin_sessions` - Admin authentication

### 4. Access Admin Panel

URL: `https://license.strikesnake.com/admin/`

**Default credentials:**
- Username: `admin`
- Password: `admin123`

⚠️ **IMPORTANT:** Change password immediately after first login!

## 📁 File Structure

```
license-system/
├── config.php              # Database & security config
├── install.php             # Database installer
├── README.md               # This file
│
├── api/
│   ├── validate.php        # License validation endpoint
│   └── deactivate.php      # License deactivation endpoint
│
└── admin/
    ├── index.php           # Dashboard & license management
    ├── login.php           # Admin login
    ├── logout.php          # Admin logout
    └── style.css           # Admin panel styles
```

## 🔌 API Endpoints

### 1. Validate License

**Endpoint:** `POST https://license.strikesnake.com/api/validate.php`

**Request:**
```json
{
  "license_key": "XXXX-XXXX-XXXX-XXXX",
  "domain": "example.com",
  "product": "elementor-page-builder-assistant"
}
```

**Response (Success):**
```json
{
  "valid": true,
  "message": "License is valid",
  "license_key": "XXXX-XXXX-XXXX-XXXX",
  "customer_name": "John Doe",
  "expires_at": null,
  "status": "active",
  "domain": "example.com"
}
```

**Response (Error):**
```json
{
  "valid": false,
  "message": "License is already activated on another domain",
  "activated_domain": "other-site.com"
}
```

### 2. Deactivate License

**Endpoint:** `POST https://license.strikesnake.com/api/deactivate.php`

**Request:**
```json
{
  "license_key": "XXXX-XXXX-XXXX-XXXX",
  "domain": "example.com"
}
```

**Response:**
```json
{
  "success": true,
  "message": "License deactivated successfully"
}
```

## 💻 Usage

### Creating a License

1. Login to admin panel
2. Fill customer information
3. Set expiration date (optional - leave empty for lifetime)
4. Click "Generate License"
5. Copy license key and send to customer

### Managing Licenses

- **Change Status:** Use dropdown in Actions column
- **Reset Domain:** Click 🔄 to unbind domain (allows customer to use on different site)
- **Delete License:** Click 🗑️ to permanently remove
- **View Logs:** Click 📊 to see activity history

### License Statuses

- **Active** ✅ - License is valid and can be used
- **Inactive** ⚠️ - License exists but cannot be activated
- **Suspended** 🚫 - Temporarily disabled (e.g., payment issue)
- **Expired** ⏱️ - Expiration date passed (auto-set by system)

## 🔒 Security Features

1. **Password Hashing** - BCrypt hashing for admin passwords
2. **SQL Injection Protection** - Prepared statements
3. **Session Management** - Secure token-based sessions
4. **CORS Headers** - Controlled API access
5. **Input Validation** - All inputs sanitized
6. **Activity Logging** - Track suspicious activity

## 🌐 Integration with WordPress Plugin

Update `includes/class-license.php` in your WordPress plugin:

```php
function ebi_validate_license_remote( $license_key ) {
    $api_url = 'https://license.strikesnake.com/api/validate.php';
    
    $site_url = get_site_url();
    $domain = parse_url( $site_url, PHP_URL_HOST );
    
    $response = wp_remote_post( $api_url, array(
        'timeout' => 15,
        'body' => array(
            'license_key' => $license_key,
            'domain' => $domain,
            'product' => 'elementor-page-builder-assistant',
        ),
    ) );
    
    if ( is_wp_error( $response ) ) {
        return ebi_check_grace_period();
    }
    
    $body = wp_remote_retrieve_body( $response );
    $data = json_decode( $body, true );
    
    if ( isset( $data['valid'] ) && $data['valid'] === true ) {
        update_option( 'ebi_license_last_check', time() );
        return true;
    }
    
    return false;
}
```

## 📊 Database Schema

### licenses table
- `id` - Primary key
- `license_key` - 19-char license (XXXX-XXXX-XXXX-XXXX)
- `customer_name` - Customer full name
- `customer_email` - Customer email
- `customer_telegram` - Telegram username
- `status` - active/inactive/suspended/expired
- `domain` - Activated domain (NULL if not activated)
- `product` - Product identifier
- `max_activations` - Max domains allowed (default: 1)
- `current_activations` - Current active domains
- `expires_at` - Expiration date (NULL for lifetime)
- `created_at` - Creation timestamp
- `updated_at` - Last update timestamp
- `notes` - Admin notes

### license_logs table
- `id` - Primary key
- `license_id` - Foreign key to licenses
- `action` - Action type (validation_success, validation_failed, first_activation, etc.)
- `ip_address` - Client IP
- `details` - JSON details
- `created_at` - Log timestamp

## 🛠️ Troubleshooting

### Database Connection Error
- Check DB credentials in `config.php`
- Verify MySQL service is running
- Ensure database exists

### API Returns 500 Error
- Check PHP error log
- Verify `config.php` is properly configured
- Ensure database tables exist

### License Validation Fails
- Check API URL is accessible
- Verify CORS headers allow requests
- Check license status in admin panel

## 📝 License Key Format

Format: `XXXX-XXXX-XXXX-XXXX` (19 characters)
- 4 groups of 4 uppercase hexadecimal characters
- Separated by hyphens
- Example: `A1B2-C3D4-E5F6-G7H8`

## 🔄 Updates & Maintenance

### Clean Old Sessions
Run periodically to clean expired sessions:

```sql
DELETE FROM admin_sessions WHERE expires_at < NOW();
```

### Clean Old Logs
Keep logs for 90 days:

```sql
DELETE FROM license_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY);
```

## 📞 Support

- Website: https://bayramsavluk.com
- Telegram: @bayramsavluk
- Email: info@bayramsavluk.com

## 📄 License

Proprietary - All rights reserved
© 2025 Bayram Şavluk
